﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;

public enum ControllerLegendButtons
{
    A,
    B,
    X,
    Y
}

public class LegendManager : MonoBehaviour
{
    public class ControllerButton
    {
        public LegendAnimations.LegendButtonState ButtonState
        {
            get
            {
                return _Animations.ButtonState;
            }
        }

        private ControllerLegendButtons _ID;
        private Image _Image;
        private TMP_Text _Text;
        private LegendAnimations _Animations;

        public ControllerButton(ControllerLegendButtons buttonID, GameObject legendGameObject)
        {
            //Set variables, get components
            _ID = buttonID;
            _Image = legendGameObject.FindChild("Image").GetComponent<Image>();
            _Text = legendGameObject.FindChild("LegendText").GetComponent<TMP_Text>();
            _Animations = legendGameObject.GetComponent<LegendAnimations>();
        }

        /// <summary>
        /// Hides the legend button
        /// </summary>
        public void Hide()
        {
            if (Instance != null && _Animations.ButtonState != LegendAnimations.LegendButtonState.Hidden)
            {
                _Animations.SetButtonState(LegendAnimations.LegendButtonState.Hiding);
            }

            _Text.text = "";
        }

        /// <summary>
        /// Shows the legend button
        /// </summary>
        /// <param name="showIfNoText">A boolean to show the legend button even if it contains no text</param>
        public void Show(bool showIfNoText = true)
        {
            if (Instance != null && (showIfNoText || (!showIfNoText && _Text.text != "")))
            {
                if (_Animations.ButtonState != LegendAnimations.LegendButtonState.Idle)
                {
                    _Animations.SetButtonState(LegendAnimations.LegendButtonState.Start);
                }
            }
        }

        /// <summary>
        /// Performs the legend button's press animation
        /// </summary>
        public void Press()
        {
            if (Instance != null && _Animations.ButtonState == LegendAnimations.LegendButtonState.Idle)
            {
                _Animations.SetButtonState(LegendAnimations.LegendButtonState.Pressing);
            }
        }

        /// <summary>
        /// Sets the text of the legend button
        /// </summary>
        /// <param name="text">The new text to display</param>
        public void SetText(string text)
        {
            _Text.SetText(text);
        }
    }

    public GameObject UILegend;

    private static LegendManager _Instance;

    public Dictionary<ControllerLegendButtons, ControllerButton> ControllerButtons { get; private set; }    //A dictionary of the controller buttons. Key - the legend button. Value - the class for the button.

    public static LegendManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<LegendManager>();
            }

            return _Instance;
        }
    }


    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        switch (e.NewState)
        {
            case Constants.InputState.MouseAndKeyboard:
                //Switched to keyboard, hide all legends
                ControllerButtons[ControllerLegendButtons.A].Hide();
                ControllerButtons[ControllerLegendButtons.B].Hide();
                ControllerButtons[ControllerLegendButtons.X].Hide();
                ControllerButtons[ControllerLegendButtons.Y].Hide();
                break;

            case Constants.InputState.Controller:
                //Switched to controller, show legends only if they have text
                ControllerButtons[ControllerLegendButtons.A].Show(false);
                ControllerButtons[ControllerLegendButtons.B].Show(false);
                ControllerButtons[ControllerLegendButtons.X].Show(false);
                ControllerButtons[ControllerLegendButtons.Y].Show(false);
                break;
        }

    }

    private void CreateControllerButtons()
    {
        ControllerButtons = new Dictionary<ControllerLegendButtons, ControllerButton>();

        //Find all game objects, create the classes
        GameObject aButtonObj = GameObject.Find("AButton");
        ControllerButton aButton = new ControllerButton(ControllerLegendButtons.A, aButtonObj);

        GameObject bButtonObj = GameObject.Find("BButton");
        ControllerButton bButton = new ControllerButton(ControllerLegendButtons.B, bButtonObj);

        GameObject xButtonObj = GameObject.Find("XButton");
        ControllerButton xButton = new ControllerButton(ControllerLegendButtons.X, xButtonObj);

        GameObject yButtonObj = GameObject.Find("YButton");
        ControllerButton yButton = new ControllerButton(ControllerLegendButtons.Y, yButtonObj);

        //All classes made, assign them into the dictionary as appropriate
        ControllerButtons[ControllerLegendButtons.A] = aButton;
        ControllerButtons[ControllerLegendButtons.B] = bButton;
        ControllerButtons[ControllerLegendButtons.X] = xButton;
        ControllerButtons[ControllerLegendButtons.Y] = yButton;
    }

    private void Start()
    {
        CreateControllerButtons();

        //Initially hide all legend buttons
        ControllerButtons[ControllerLegendButtons.A].Hide();
        ControllerButtons[ControllerLegendButtons.B].Hide();
        ControllerButtons[ControllerLegendButtons.X].Hide();
        ControllerButtons[ControllerLegendButtons.Y].Hide();

        InputManager.Instance.InputStateChanged += OnInputStateChanged;
    }

    private void Update()
    {
        if (EventSystem.current.currentSelectedGameObject == null || EventSystem.current.currentSelectedGameObject.GetComponent<LegendDisplayer>() == null)
        {
            //TODO: This makes us lose a click, but checking for idle means click gets stuck on???
            ControllerButtons[ControllerLegendButtons.A].Hide();
            ControllerButtons[ControllerLegendButtons.B].Hide();
            ControllerButtons[ControllerLegendButtons.X].Hide();
            ControllerButtons[ControllerLegendButtons.Y].Hide();
        }

        if (Input.GetKey("joystick button 0") && ControllerButtons[ControllerLegendButtons.A].ButtonState == LegendAnimations.LegendButtonState.Idle)
        {
            ControllerButtons[ControllerLegendButtons.A].Press();
        }

        else if (Input.GetKey("joystick button 1") && ControllerButtons[ControllerLegendButtons.B].ButtonState == LegendAnimations.LegendButtonState.Idle)
        {
            ControllerButtons[ControllerLegendButtons.B].Press();
        }

        else if (Input.GetKey("joystick button 2") && ControllerButtons[ControllerLegendButtons.X].ButtonState == LegendAnimations.LegendButtonState.Idle)
        {
            ControllerButtons[ControllerLegendButtons.X].Press();
        }

        else if (Input.GetKey("joystick button 3") && ControllerButtons[ControllerLegendButtons.Y].ButtonState == LegendAnimations.LegendButtonState.Idle)
        {
            ControllerButtons[ControllerLegendButtons.Y].Press();
        }
    }
}
